using System;
using System.Xml;

using gov.va.med.vbecs.DAL.VistALink.OpenLibrary;

namespace gov.va.med.vbecs.DAL.VistALink.OpenLibrary.Messages
{
	/// <summary>
	/// Update verify code request message defined in VistALink protocol.
	/// </summary>
	public sealed class SecurityUpdateVerifyCodeRequestMessage : SecurityBaseRequestMessage
	{	
		private VistASecurityCode _oldVerifyCode;
		private VistASecurityCode _newVerifyCode;
		private VistASecurityCode _newCheckVerifyCode;

		/// <summary>
		/// VistALink message request type string. 
		/// This constant is made public to recognize message XML.
		/// </summary>
		public const string RequestType = "AV.UpdateVC";

		/// <summary>
		/// VistALink message XSD schema file name string. 		
		/// This constant is made public to recognize message XML.
		/// </summary>
		public const string XsdName = "secUpdateVerifyRequest.xsd";

		// Constants used in XML message serialization/deserialization
		private const string XMLCONSTS_OLD_VC_NODE_NAME = "oldVc";
		private const string XMLCONSTS_NEW_VC_NODE_NAME = "newVc";
		private const string XMLCONSTS_CONFIRMED_VC_NODE_NAME = "confirmedVc";

		/// <summary>
		/// Constructor specifying minimal required VistALink 
		/// update verify code security request message parameters.
		/// All parameters are required and can not be null. 
		/// </summary>
		/// <param name="oldVerifyCode">Old verify code.</param>
		/// <param name="newVerifyCode">New verify code.</param>
		/// <param name="newCheckVerifyCode">New check verify code.</param>
		public SecurityUpdateVerifyCodeRequestMessage( VistASecurityCode oldVerifyCode, VistASecurityCode newVerifyCode, VistASecurityCode newCheckVerifyCode ) 
			: base( XsdName, RequestType )
		{
			if( oldVerifyCode == null )
				throw( new ArgumentNullException( "oldVerifyCode" ) );

			if( newVerifyCode == null )
				throw( new ArgumentNullException( "newVerifyCode" ) );

			if( newCheckVerifyCode == null )
				throw( new ArgumentNullException( "newCheckVerifyCode" ) );

			_oldVerifyCode = oldVerifyCode;
			_newVerifyCode = newVerifyCode;
			_newCheckVerifyCode = newCheckVerifyCode;
		}

		/// <summary>
		/// Standard VistALink message deserialization constructor. 
		/// Allows recreating message object from XML document.
		/// </summary>
		/// <param name="sourceDoc">Source XML document.</param>
		public SecurityUpdateVerifyCodeRequestMessage( XmlDocument sourceDoc )
			: base( XsdName, RequestType, sourceDoc ) 
		{
			XmlElement _requestElement = XmlParseGetRequestElement( sourceDoc );

			_oldVerifyCode = VistASecurityCode.ParseVistAObfuscatedString( XmlUtility.ParseGetRequiredElementWithUniqueNameCDataValue( _requestElement, XMLCONSTS_OLD_VC_NODE_NAME ) );
			_newVerifyCode = VistASecurityCode.ParseVistAObfuscatedString( XmlUtility.ParseGetRequiredElementWithUniqueNameCDataValue( _requestElement, XMLCONSTS_NEW_VC_NODE_NAME ) );
			_newCheckVerifyCode = VistASecurityCode.ParseVistAObfuscatedString( XmlUtility.ParseGetRequiredElementWithUniqueNameCDataValue( _requestElement, XMLCONSTS_CONFIRMED_VC_NODE_NAME ) );
		}

		/// <summary>
		/// XML serialization method addind old, new and new check verify codes to 
		/// request element created by the base class.
		/// </summary>
		/// <param name="writer">XmlWriter to use.</param>
		protected override void WriteXmlRequestNodeContent( XmlWriter writer )
		{
			XmlUtility.GenWriteElementWithCData( writer, XMLCONSTS_OLD_VC_NODE_NAME, _oldVerifyCode.ToVistAObfuscatedString() );
			XmlUtility.GenWriteElementWithCData( writer, XMLCONSTS_NEW_VC_NODE_NAME, _newVerifyCode.ToVistAObfuscatedString() );
			XmlUtility.GenWriteElementWithCData( writer, XMLCONSTS_CONFIRMED_VC_NODE_NAME, _newCheckVerifyCode.ToVistAObfuscatedString() );
		}

		/// <summary>
		/// Old verify code. 
		/// </summary>
		public VistASecurityCode OldVerifyCode
		{
			get
			{
				return _oldVerifyCode;
			}
		}

		/// <summary>
		/// New verify code.
		/// </summary>
		public VistASecurityCode NewVerifyCode
		{
			get
			{
				return _newVerifyCode;
			}
		}

		/// <summary>
		/// Double entry for new security code - added for verification. 
		/// </summary>
		public VistASecurityCode NewCheckVerifyCode
		{
			get
			{
				return _newCheckVerifyCode;
			}
		}
	}
}
